BlocFolio — Programmatic SEO Crypto Site (PHP + MySQL)

Stack: Plain PHP (no Composer), MySQL 8, Bootstrap 5 (CDN), ECharts (CDN), DataTables (CDN)

This repo contains an MVP scaffold matching the architecture you outlined, with:
- A tiny PHP MVC (router, controllers, views)
- Coins list and coin detail routes (placeholders wired to DB)
- Core MySQL schema for coins + market snapshots
- Ingestion script stubs (CoinGecko) with rate-limit scaffolding
- File cache utilities and HTML snapshot hooks

Pages

- Coins list and coin detail with server-side DataTables, OHLC chart and metadata
- Exchanges list and detail with markets table (server-side)
- Categories list and detail with member coins (server-side)
- Analysis hub with Top Gainers/Losers (24h)

Quick start

1) Create a database and user in MySQL 8.
2) Import schema: `mysql -u USER -p DB_NAME < sql/schema.sql`
3) Copy `app/config.php` to `app/config.local.php` and edit DB credentials.
4) Serve `public/` via Apache/Nginx (Apache: use provided .htaccess).
5) Visit `/coins` — lists coins from DB (empty until ingestion).

Local test data

- Seed a few coins and snapshots: `php scripts/seed/sample_data.php`
- Then refresh `/coins` and click into a coin.

Scripts

- `scripts/ingest/markets_pull.php` — pulls `/coins/markets` snapshots (curl-based, with simple backoff)
- `scripts/ingest/ohlc_sync.php` — placeholder for OHLC sync (wire like markets)
- `scripts/ingest/metadata_refresh.php` — pulls `/coins/{id}` details into `coin_metadata`
- `scripts/seo/sitemap_builder.php` — builds `/public/sitemaps/*` and index (includes categories)
- `scripts/ingest/exchanges_pull.php` — exchanges + top tickers (first page) ingest
- `scripts/ingest/categories_pull.php` — categories ingest (stub mapping)
- `scripts/ingest/coin_categories_sync.php` — populates coin↔category mapping from per-coin metadata
- Add crons per `scripts/cron.sample.txt`

Production checklist

- PHP extensions: `pdo_mysql`, `curl`, `dom`, `json`, `mbstring`, `openssl`
- Web server: point docroot to `public/`; enable URL rewrite (Apache `.htaccess` or Nginx try_files)
- Env: set `APP_ENV=prod` and `app.base_url` in `app/config.local.php`
- DB: import `sql/schema.sql`; create a dedicated MySQL user with least privilege
- Cron: configure jobs from `scripts/cron.sample.txt` with your PHP path
- Logs: pipe cron output to log files; monitor for 429s and adjust `coingecko.rate_limit_per_min`
- Security: set strong `open_basedir`, disable `display_errors` in prod, enable HTTPS + HSTS
- Caching: Cloudflare or server cache for `/sitemaps/*` and static assets
- Permissions: web/PHP user must have write access to `storage/` (subfolders: `cache`, `html_snapshots`, `locks`)

Admin settings

- Token-gated route: `/admin/settings?token=YOUR_TOKEN` — set in `app/config.local.php` as `admin_token`
- Stores overrides in DB table `admin_settings` and overlays at boot

Sitemap pings

- `scripts/seo/sitemap_ping.php` hits Google/Bing ping endpoints (requires `app.base_url`)

Operations

- Ingestion scripts use file locks to avoid concurrent runs (`storage/locks`)
- HTTP backoff: handles 429 with `Retry-After`; exponential backoff on errors
- Snapshots: coin pages write snapshots in prod and are invalidated on data updates
- Sitemaps: coin and category sitemaps generated under `public/sitemaps/`

Nginx snippet

```
server {
  listen 80;
  server_name example.com;
  root /var/www/blocfolio/public;
  index index.php;

  location /sitemaps/ { try_files $uri =404; }
  location / { try_files $uri $uri/ /index.php?$args; }
  location ~ \.php$ {
    include fastcgi_params;
    fastcgi_pass unix:/run/php/php8.2-fpm.sock;
    fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
  }
}
```

Server-side DataTables

- Endpoint: `/api/coins` accepts standard DataTables params (`draw,start,length,order,search`)
- The `/coins` page now uses server-side processing for scalability.

SEO & Snapshots

- Coin pages include meta/OG and JSON-LD placeholders; canonical set from `app.base_url`.
- In `prod`, coin pages write HTML snapshots to `storage/html_snapshots/` on render.
- Bots get pre-rendered snapshots when available (served from front controller).
- Dynamic robots: `/robots.txt` responds with `Sitemap: {base}/sitemaps/sitemap-index.xml`.

Snapshot maintenance

- Purge all: `php scripts/seo/snapshots_purge.php`
- Purge a route: `php scripts/seo/snapshots_purge.php /coins/bitcoin`
- Snapshots auto-invalidated by ingestion scripts after updates.

OHLC API

- `/api/coins/{slug}/ohlc?interval=1d&limit=365&currency=usd` returns `[timestamp, o,h,l,c,v]` series from DB.

Sitemaps

- Build with: `php scripts/seo/sitemap_builder.php` (ensure `app.base_url` set)
- Access index: `/sitemaps/sitemap-index.xml`

Nav & routes

- `/coins`, `/coins/{slug}`
- `/exchanges`, `/exchanges/{slug}`
- `/categories`, `/categories/{slug}`
- `/analysis`

Notes

- No Composer used. Keep dependencies via CDN.
- For Nginx, route all unknown paths to `public/index.php`.
- In production, set `APP_ENV=prod` and configure caching and HTML snapshots.
