<?php
namespace App\Controllers;

use App\Database;
use App\Config;
use App\View;

class AnalysisController
{
    public function index(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        $limit = 15;

        $gainers = $this->topChange($pdo, $currency, 'DESC', $limit);
        $losers  = $this->topChange($pdo, $currency, 'ASC',  $limit);

        echo View::render('analysis/index', [
            'title' => 'Analysis — BlocFolio',
            'gainers' => $gainers,
            'losers' => $losers,
        ]);
    }

    private function topChange($pdo, string $currency, string $order, int $limit): array
    {
        $sql = "
            SELECT c.slug, c.name, c.symbol, c.rank,
                   ms.price, ms.change_24h, ms.market_cap
            FROM coins c
            JOIN coin_market_snapshots ms ON ms.id = (
                SELECT id FROM coin_market_snapshots
                WHERE coin_id = c.id AND currency = :currency
                ORDER BY timestamp DESC LIMIT 1
            )
            WHERE c.is_active = 1 AND ms.market_cap IS NOT NULL AND ms.volume_24h IS NOT NULL AND ms.change_24h IS NOT NULL
            ORDER BY ms.change_24h $order
            LIMIT :limit
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':currency', $currency);
        $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // API: heatmap data
    public function heatmap(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        $sql = "
            SELECT c.slug, c.symbol, c.name, ms.market_cap, ms.change_24h
            FROM coins c
            JOIN coin_market_snapshots ms ON ms.id = (
                SELECT id FROM coin_market_snapshots WHERE coin_id = c.id AND currency = :currency ORDER BY timestamp DESC LIMIT 1
            )
            WHERE c.is_active = 1 AND ms.market_cap IS NOT NULL
            ORDER BY ms.market_cap DESC
            LIMIT 200
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':currency'=>$currency]);
        header('Content-Type: application/json');
        echo json_encode(['data'=>$stmt->fetchAll()]);
    }

    public function breadth(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        $sql = "
            SELECT 
              SUM(ms.change_24h > 0) AS adv,
              SUM(ms.change_24h <= 0 OR ms.change_24h IS NULL) AS dec,
              COUNT(*) AS total
            FROM coins c
            JOIN coin_market_snapshots ms ON ms.id = (
                SELECT id FROM coin_market_snapshots WHERE coin_id = c.id AND currency = :currency ORDER BY timestamp DESC LIMIT 1
            )
            WHERE c.is_active = 1 AND ms.market_cap IS NOT NULL
            ORDER BY ms.market_cap DESC
            LIMIT 200
        ";
        $st = $pdo->prepare($sql);
        $st->execute([':currency'=>$currency]);
        $r = $st->fetch();
        $adv = (int)($r['adv'] ?? 0); $dec = (int)($r['dec'] ?? 0); $total = (int)($r['total'] ?? 0);
        $pct = $total ? round(($adv/$total)*100, 2) : 0;
        header('Content-Type: application/json'); echo json_encode(['adv'=>$adv,'dec'=>$dec,'total'=>$total,'pct_adv'=>$pct]);
    }

    public function highsLows(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        // Daily close equals max/min close in last 365 days
        $sqlHighs = "
          SELECT c.slug, c.name, c.symbol, o.c as close
          FROM coins c
          JOIN coin_ohlc_1d o ON o.coin_id = c.id AND o.currency = :currency
          WHERE o.t = (SELECT MAX(t) FROM coin_ohlc_1d WHERE coin_id = c.id AND currency = :currency)
            AND o.c IS NOT NULL
            AND o.c >= (SELECT MAX(c) FROM coin_ohlc_1d WHERE coin_id = c.id AND currency = :currency AND t >= DATE_SUB(NOW(), INTERVAL 365 DAY))
          ORDER BY o.c DESC LIMIT 50";
        $stH = $pdo->prepare($sqlHighs); $stH->execute([':currency'=>$currency]);
        $highs = $stH->fetchAll();
        $sqlLows = "
          SELECT c.slug, c.name, c.symbol, o.c as close
          FROM coins c
          JOIN coin_ohlc_1d o ON o.coin_id = c.id AND o.currency = :currency
          WHERE o.t = (SELECT MAX(t) FROM coin_ohlc_1d WHERE coin_id = c.id AND currency = :currency)
            AND o.c IS NOT NULL
            AND o.c <= (SELECT MIN(c) FROM coin_ohlc_1d WHERE coin_id = c.id AND currency = :currency AND t >= DATE_SUB(NOW(), INTERVAL 365 DAY))
          ORDER BY o.c ASC LIMIT 50";
        $stL = $pdo->prepare($sqlLows); $stL->execute([':currency'=>$currency]);
        $lows = $stL->fetchAll();
        header('Content-Type: application/json'); echo json_encode(['highs'=>$highs, 'lows'=>$lows]);
    }

    public function volumeSurge(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        // Compare latest volume_24h to average of last 24h snapshots
        $sql = "
          SELECT c.slug, c.name, c.symbol,
                 ms.volume_24h AS vol,
                 (SELECT AVG(s.volume_24h) FROM coin_market_snapshots s WHERE s.coin_id = c.id AND s.currency = :currency AND s.timestamp >= DATE_SUB(NOW(), INTERVAL 24 HOUR)) AS avg_vol
          FROM coins c
          JOIN coin_market_snapshots ms ON ms.id = (
              SELECT id FROM coin_market_snapshots WHERE coin_id = c.id AND currency = :currency ORDER BY timestamp DESC LIMIT 1
          )
          WHERE c.is_active = 1 AND ms.volume_24h IS NOT NULL
        ";
        $st = $pdo->prepare($sql); $st->execute([':currency'=>$currency]);
        $rows = [];
        foreach ($st->fetchAll() as $r) {
            $avg = (float)($r['avg_vol'] ?? 0); $cur = (float)($r['vol'] ?? 0);
            if ($avg > 0 && $cur > 0) { $r['ratio'] = $cur / $avg; $rows[] = $r; }
        }
        usort($rows, fn($a,$b)=>($b['ratio']<=>$a['ratio']));
        $rows = array_slice($rows, 0, 50);
        header('Content-Type: application/json'); echo json_encode(['data'=>$rows]);
    }

    public function volatility(): void
    {
        $pdo = Database::pdo();
        $currency = strtolower(Config::get('app.currency_default','USD'));
        $sql = "
          SELECT c.slug, c.name, c.symbol, t.stddev
          FROM coins c
          JOIN (
            SELECT coin_id, STDDEV_POP(c) AS stddev
            FROM coin_ohlc_1d WHERE currency = :currency AND t >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY coin_id
          ) t ON t.coin_id = c.id
          ORDER BY t.stddev DESC
          LIMIT 50
        ";
        $st = $pdo->prepare($sql); $st->execute([':currency'=>$currency]);
        header('Content-Type: application/json'); echo json_encode(['data'=>$st->fetchAll()]);
    }
}
