<?php
namespace App\Models;

use App\Database;
use App\Config;
use PDO;

class Coin
{
    public static function paginated(int $page, int $limit, string $sort, string $dir, array $filters): array
    {
        $pdo = Database::pdo();
        $offset = ($page - 1) * $limit;
        $currency = Config::get('app.currency_default', 'USD');

        // Map sort keys to SQL expressions
        $sortMap = [
            'rank' => 'c.rank',
            'name' => 'c.name',
            'price' => 'ms.price',
            'market_cap' => 'ms.market_cap',
            'volume_24h' => 'ms.volume_24h',
            'change_24h' => 'ms.change_24h',
        ];
        $sortCol = $sortMap[$sort] ?? 'c.rank';
        $dir = strtoupper($dir) === 'DESC' ? 'DESC' : 'ASC';

        $where = ['c.is_active = 1'];
        $params = [':currency' => strtolower($currency)];

        if (!empty($filters['q'])) {
            $where[] = '(c.name LIKE :q OR c.symbol LIKE :q)';
            $params[':q'] = '%' . $filters['q'] . '%';
        }
        if (!empty($filters['category'])) {
            $where[] = 'c.category_primary = :category';
            $params[':category'] = $filters['category'];
        }
        $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

        // Count
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM coins c $whereSql");
        $stmt->execute(array_diff_key($params, [':currency' => true]));
        $total = (int)$stmt->fetchColumn();

        // Data query with latest market snapshot per coin for currency
        $sql = "
            SELECT c.id, c.cg_id, c.symbol, c.name, c.slug, c.image_small, c.rank,
                   ms.price, ms.market_cap, ms.volume_24h, ms.change_24h
            FROM coins c
            LEFT JOIN coin_market_snapshots ms
              ON ms.id = (
                  SELECT id FROM coin_market_snapshots
                  WHERE coin_id = c.id AND currency = :currency
                  ORDER BY timestamp DESC LIMIT 1
              )
            $whereSql
            ORDER BY $sortCol $dir
            LIMIT :limit OFFSET :offset
        ";
        $stmt = $pdo->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        return [$rows, $total];
    }

    public static function findBySlug(string $slug): ?array
    {
        $pdo = Database::pdo();
        $currency = Config::get('app.currency_default', 'USD');
        $sql = "
            SELECT c.*, cm.description_html,
                   ms.price, ms.market_cap, ms.volume_24h, ms.change_1h, ms.change_24h, ms.change_7d, ms.change_30d,
                   ms.ath_drawdown, ms.timestamp as snapshot_at
            FROM coins c
            LEFT JOIN coin_metadata cm ON cm.coin_id = c.id
            LEFT JOIN coin_market_snapshots ms
              ON ms.id = (
                  SELECT id FROM coin_market_snapshots
                  WHERE coin_id = c.id AND currency = :currency
                  ORDER BY timestamp DESC LIMIT 1
              )
            WHERE c.slug = :slug
            LIMIT 1
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':slug' => $slug,
            ':currency' => strtolower($currency),
        ]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public static function idBySlug(string $slug): ?int
    {
        $pdo = Database::pdo();
        $stmt = $pdo->prepare('SELECT id FROM coins WHERE slug = :slug LIMIT 1');
        $stmt->execute([':slug' => $slug]);
        $id = $stmt->fetchColumn();
        return $id ? (int)$id : null;
    }

    public static function dataTable(int $start, int $length, string $sort, string $dir, ?string $search): array
    {
        $pdo = Database::pdo();
        $currency = Config::get('app.currency_default', 'USD');
        $sortMap = [
            'rank' => 'c.rank',
            'name' => 'c.name',
            'price' => 'ms.price',
            'market_cap' => 'ms.market_cap',
            'volume_24h' => 'ms.volume_24h',
            'change_24h' => 'ms.change_24h',
        ];
        $sortCol = $sortMap[$sort] ?? 'c.rank';
        $dir = strtoupper($dir) === 'DESC' ? 'DESC' : 'ASC';
        $baseWhere = 'c.is_active = 1';

        $total = (int)$pdo->query('SELECT COUNT(*) FROM coins c WHERE ' . $baseWhere)->fetchColumn();

        $params = [':currency' => strtolower($currency)];
        $where = $baseWhere;
        if ($search) {
            $where .= ' AND (c.name LIKE :q OR c.symbol LIKE :q)';
            $params[':q'] = '%' . $search . '%';
        }
        $stmt = $pdo->prepare('SELECT COUNT(*) FROM coins c WHERE ' . $where);
        $stmt->execute(array_diff_key($params, [':currency'=>true]));
        $filtered = (int)$stmt->fetchColumn();

        $sql = "
            SELECT c.rank, c.name, c.slug, c.symbol, c.image_small,
                   ms.price, ms.change_24h, ms.market_cap, ms.volume_24h
            FROM coins c
            LEFT JOIN coin_market_snapshots ms
              ON ms.id = (
                  SELECT id FROM coin_market_snapshots
                  WHERE coin_id = c.id AND currency = :currency
                  ORDER BY timestamp DESC LIMIT 1
              )
            WHERE $where
            ORDER BY $sortCol $dir
            LIMIT :length OFFSET :start
        ";
        $stmt = $pdo->prepare($sql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->bindValue(':length', $length, PDO::PARAM_INT);
        $stmt->bindValue(':start', $start, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        return [$rows, $total, $filtered];
    }
}
