<?php
// Front controller
require __DIR__ . '/../app/bootstrap.php';

// Serve HTML snapshots to bots if available and fresh
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET') {
    $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
    $isBot = preg_match('/googlebot|bingbot|duckduckbot|baiduspider|yandex|twitterbot|facebookexternalhit|linkedinbot|slackbot|ahrefs|semrush/i', $ua);
    $path = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
    if ($isBot && $path) {
        $safe = trim($path, '/');
        if ($safe === '') $safe = 'home';
        $safe = preg_replace('/[^a-zA-Z0-9_\-\/]/', '_', $safe);
        $safe = str_replace('/', '__', $safe);
        $file = __DIR__ . '/../storage/html_snapshots/' . $safe . '.html';
        if (is_file($file)) {
            $html = file_get_contents($file);
            if (preg_match('/etag:([a-f0-9]{40}).*?expires:(\d+)/', $html, $m)) {
                $etag = $m[1];
                $expiresAt = (int)$m[2];
                if ($expiresAt > time()) {
                    if (!empty($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH']) === $etag) {
                        header('ETag: ' . $etag);
                        header('Cache-Control: public, max-age=300');
                        http_response_code(304);
                        exit;
                    }
                    header('ETag: ' . $etag);
                    header('Cache-Control: public, max-age=300');
                    header('Last-Modified: ' . gmdate('D, d M Y H:i:s', filemtime($file)) . ' GMT');
                    header('Content-Type: text/html; charset=utf-8');
                    echo $html;
                    exit;
                }
            }
        }
    }
}

// Routing
$router = new App\Router();

$router->get('/', function () {
    header('Location: /coins');
    exit;
});

$router->get('/coins', [App\Controllers\CoinsController::class, 'index']);
$router->get('/coins/{slug}', [App\Controllers\CoinsController::class, 'show']);
// API
$router->get('/api/coins', [App\Controllers\Api\CoinsApiController::class, 'index']);
$router->get('/api/coins/{slug}/ohlc', [App\Controllers\Api\OHLCController::class, 'bySlug']);

// Exchanges
$router->get('/exchanges', [App\Controllers\ExchangesController::class, 'index']);
$router->get('/exchanges/{slug}', [App\Controllers\ExchangesController::class, 'show']);
$router->get('/api/exchanges', [App\Controllers\Api\ExchangesApiController::class, 'list']);
$router->get('/api/exchanges/{slug}/markets', [App\Controllers\Api\ExchangesApiController::class, 'markets']);

// Categories
$router->get('/categories', [App\Controllers\CategoriesController::class, 'index']);
$router->get('/categories/{slug}', [App\Controllers\CategoriesController::class, 'show']);
$router->get('/api/categories', [App\Controllers\Api\CategoriesApiController::class, 'list']);
$router->get('/api/categories/{slug}/coins', [App\Controllers\Api\CategoriesApiController::class, 'coins']);

// Analysis
$router->get('/analysis', [App\\Controllers\\AnalysisController::class, 'index']);
$router->get('/api/analysis/heatmap', [App\\Controllers\\AnalysisController::class, 'heatmap']);
$router->get('/api/analysis/breadth', [App\\Controllers\\AnalysisController::class, 'breadth']);
$router->get('/api/analysis/highs-lows', [App\\Controllers\\AnalysisController::class, 'highsLows']);
$router->get('/api/analysis/volume-surge', [App\\Controllers\\AnalysisController::class, 'volumeSurge']);
$router->get('/api/analysis/volatility', [App\\Controllers\\AnalysisController::class, 'volatility']);

// Admin settings (token-gated)
$router->get('/admin/settings', [App\\Controllers\\AdminSettingsController::class, 'index']);
$router->post('/admin/settings', [App\\Controllers\\AdminSettingsController::class, 'save']);
$router->get('/admin/jobs', [App\\Controllers\\AdminJobsController::class, 'index']);
$router->post('/admin/jobs/enqueue', [App\\Controllers\\AdminJobsController::class, 'enqueue']);

// Auth
$router->get('/login', [App\\Controllers\\AuthController::class, 'loginForm']);
$router->post('/login', [App\\Controllers\\AuthController::class, 'loginPost']);
$router->get('/register', [App\\Controllers\\AuthController::class, 'registerForm']);
$router->post('/register', [App\\Controllers\\AuthController::class, 'registerPost']);
$router->get('/logout', [App\\Controllers\\AuthController::class, 'logout']);

// Tools
$router->get('/tools', [App\\Controllers\\ToolsController::class, 'index']);
$router->get('/tools/pl', [App\\Controllers\\ToolsController::class, 'pl']);
$router->get('/tools/cagr', [App\\Controllers\\ToolsController::class, 'cagr']);
$router->get('/tools/dca', [App\\Controllers\\ToolsController::class, 'dca']);
$router->get('/tools/position', [App\\Controllers\\ToolsController::class, 'position']);

// Compare
$router->get('/compare/{pair}', [App\\Controllers\\CompareController::class, 'show']);

// Alerts
$router->get('/alerts', [App\\Controllers\\AlertsController::class, 'index']);
$router->post('/alerts/create', [App\\Controllers\\AlertsController::class, 'create']);
$router->post('/alerts/delete', [App\\Controllers\\AlertsController::class, 'delete']);

// Screens
$router->get('/screens', [App\\Controllers\\ScreensController::class, 'index']);
$router->get('/screens/new', [App\\Controllers\\ScreensController::class, 'createForm']);
$router->post('/screens/new', [App\\Controllers\\ScreensController::class, 'create']);
$router->get('/screens/{slug}', [App\\Controllers\\ScreensController::class, 'show']);
$router->get('/api/screens/{slug}/csv', [App\\Controllers\\ScreensController::class, 'exportCsv']);
$router->get('/api/screens/preview', [App\\Controllers\\ScreensController::class, 'preview']);
$router->post('/api/screens/{slug}/star', [App\\Controllers\\Api\\ScreensApiController::class, 'star']);

// Username namespace for screens (resolve by slug; username for aesthetics)
$router->get('/screens/{username}/{slug}', [App\\Controllers\\ScreensController::class, 'show']);

// Watchlists (owner)
$router->get('/watchlists', [App\\Controllers\\WatchlistsController::class, 'index']);
$router->post('/watchlists/new', [App\\Controllers\\WatchlistsController::class, 'create']);
// Public watchlist route
$router->get('/u/{username}/watchlists/{slug}', [App\\Controllers\\WatchlistsController::class, 'showPublic']);
$router->get('/u/{username}/watchlists/{slug}.csv', [App\\Controllers\\WatchlistsController::class, 'exportCsv']);

// Watchlist API
$router->post('/api/watchlist/toggle', [App\\Controllers\\Api\\WatchlistApiController::class, 'toggle']);
$router->get('/api/watchlist/membership', [App\\Controllers\\Api\\WatchlistApiController::class, 'membership']);
$router->get('/api/watchlists', [App\\Controllers\\Api\\WatchlistApiController::class, 'listUserWatchlists']);

// robots.txt (dynamic to include sitemap index)
$router->get('/robots.txt', function () {
    $base = rtrim(App\Config::get('app.base_url', ''), '/');
    $sitemap = $base . '/sitemaps/sitemap-index.xml';
    header('Content-Type: text/plain');
    echo "User-agent: *\n";
    echo "Allow: /\n\n";
    echo "Sitemap: $sitemap\n";
});

// 404 fallback

$router->fallback(function () {
    http_response_code(404);
    echo App\View::render('errors/404', ['title' => 'Not Found']);
});

$router->dispatch($_SERVER['REQUEST_METHOD'], $_SERVER['REQUEST_URI']);
