<?php
// Pull exchanges list and markets from CoinGecko
require __DIR__ . '/../../app/bootstrap.php';
require_once __DIR__ . '/../lib/http.php';
require_once __DIR__ . '/../lib/lock.php';

use App\Database;
use App\Utils;
use App\Snapshot;
use Scripts\Lib\Http;
use Scripts\Lib\HttpException;
use Scripts\Lib\Lock;

if (php_sapi_name() !== 'cli') { echo "Run from CLI\n"; exit(1); }
set_time_limit(0);
ini_set('memory_limit','512M');

$lock = new Lock('exchanges_pull');
if (!$lock->acquire()) { echo "Another exchanges_pull is running.\n"; exit(0); }

$pdo = Database::pdo();
$base = 'https://api.coingecko.com/api/v3';

$upsertEx = $pdo->prepare("INSERT INTO exchanges (cg_id, name, slug, year_established, country, trust_score, url, logo)
VALUES (:cg_id,:name,:slug,:year,:country,:trust,:url,:logo)
ON DUPLICATE KEY UPDATE name=VALUES(name), slug=VALUES(slug), year_established=VALUES(year_established), country=VALUES(country), trust_score=VALUES(trust_score), url=VALUES(url), logo=VALUES(logo), updated_at=NOW()");

// Markets insert
$insMkt = $pdo->prepare("INSERT INTO exchange_markets (exchange_id, base_symbol, quote_symbol, pair, last_price, volume_24h, is_spot)
VALUES (:ex,:base,:quote,:pair,:last,:vol,:spot)");

try {
    $list = Http::getJson($base . '/exchanges?per_page=250&page=1', 30);
} catch (HttpException $e) {
    if ($e->getStatus()===429 && $e->getRetryAfter()) sleep($e->getRetryAfter());
    fwrite(STDERR, 'HTTP error: ' . $e->getMessage() . "\n");
    $list = [];
} catch (\Throwable $e) {
    fwrite(STDERR, 'HTTP error: ' . $e->getMessage() . "\n");
    $list = [];
}

foreach ($list as $ex) {
    $cg = $ex['id'] ?? null; if (!$cg) continue;
    $slug = Utils::slug($ex['id']);
    $upsertEx->execute([
        ':cg_id'=>$cg, ':name'=>$ex['name'] ?? $cg, ':slug'=>$slug,
        ':year'=>$ex['year_established'] ?? null, ':country'=>$ex['country'] ?? null,
        ':trust'=>$ex['trust_score'] ?? null, ':url'=>$ex['url'] ?? null, ':logo'=>$ex['image'] ?? null,
    ]);
    // Get DB id
    $id = (int)$pdo->query("SELECT id FROM exchanges WHERE cg_id = " . $pdo->quote($cg))->fetchColumn();
    if (!$id) continue;

    // Fetch markets (first page only in this stub)
    try {
        $markets = Http::getJson($base . '/exchanges/' . rawurlencode($cg) . '/tickers?per_page=100&page=1', 30);
        $tickers = $markets['tickers'] ?? [];
    } catch (HttpException $e) { if ($e->getStatus()===429 && $e->getRetryAfter()) sleep($e->getRetryAfter()); $tickers = []; }
      catch (\Throwable $e) { $tickers = []; }

    $pdo->beginTransaction();
    try {
        // Clear existing (simple approach for stub)
        $pdo->prepare('DELETE FROM exchange_markets WHERE exchange_id = :ex')->execute([':ex'=>$id]);
        foreach ($tickers as $t) {
            $baseS = $t['base'] ?? null; $quoteS = $t['target'] ?? null; if (!$baseS || !$quoteS) continue;
            $pair = $baseS . '/' . $quoteS;
            $insMkt->execute([
                ':ex'=>$id, ':base'=>$baseS, ':quote'=>$quoteS, ':pair'=>$pair,
                ':last'=>$t['last'] ?? null, ':vol'=>$t['volume'] ?? null,
                ':spot'=> isset($t['market']['type']) ? (int)($t['market']['type'] === 'spot') : 1,
            ]);
        }
        $pdo->commit();
    } catch (\Throwable $e) {
        $pdo->rollBack();
        fwrite(STDERR, 'DB error: ' . $e->getMessage() . "\n");
    }

    // Invalidate exchange snapshot
    Snapshot::invalidate('/exchanges/' . $slug);
}

// Invalidate exchanges list snapshot
\App\Snapshot::invalidate('/exchanges');

echo "exchanges_pull done\n";
