<?php
// Refresh coin metadata (description, links) from CoinGecko /coins/{id}
require __DIR__ . '/../../app/bootstrap.php';
require_once __DIR__ . '/../lib/http.php';
require_once __DIR__ . '/../lib/lock.php';

use App\Database;
use Scripts\Lib\Http;
use Scripts\Lib\HttpException;
use Scripts\Lib\Lock;
use App\Snapshot;

if (php_sapi_name() !== 'cli') { echo "Run from CLI\n"; exit(1); }
set_time_limit(0);
ini_set('memory_limit','512M');

$lock = new Lock('metadata_refresh');
if (!$lock->acquire()) { echo "Another metadata_refresh is running.\n"; exit(0); }

$pdo = Database::pdo();
$coinsStmt = $pdo->query("SELECT id, cg_id FROM coins WHERE is_active = 1 AND cg_id IS NOT NULL ORDER BY rank IS NULL, rank ASC LIMIT 200");
$coins = $coinsStmt->fetchAll();

$upsert = $pdo->prepare("REPLACE INTO coin_metadata (coin_id, description_html, homepage, explorers_json, community_json, dev_json, supply_json, hashing_algo, links_json) VALUES (:coin_id, :description_html, :homepage, :explorers_json, :community_json, :dev_json, :supply_json, :hashing_algo, :links_json)");

foreach ($coins as $coin) {
    $cgId = $coin['cg_id'];
    $url = 'https://api.coingecko.com/api/v3/coins/' . urlencode($cgId);
    echo "Fetching metadata for {$cgId}...\n";
    try {
        $data = Http::getJson($url, 30);
    } catch (HttpException $e) {
        if ($e->getStatus() === 429 && $e->getRetryAfter()) { sleep($e->getRetryAfter()); }
        fwrite(STDERR, "HTTP error for {$cgId}: " . $e->getMessage() . "\n");
        continue;
    } catch (\Throwable $e) {
        fwrite(STDERR, "HTTP error for {$cgId}: " . $e->getMessage() . "\n");
        continue;
    }
    $desc = $data['description']['en'] ?? null;
    $homepage = $data['links']['homepage'][0] ?? null;
    $explorers = $data['links']['blockchain_site'] ?? [];
    $community = [
        'twitter' => $data['links']['twitter_screen_name'] ?? null,
        'reddit' => $data['links']['subreddit_url'] ?? null,
        'telegram' => $data['links']['telegram_channel_identifier'] ?? null,
    ];
    $dev = $data['developer_data'] ?? null;
    $supply = [
        'circulating' => $data['market_data']['circulating_supply'] ?? null,
        'total' => $data['market_data']['total_supply'] ?? null,
        'max' => $data['market_data']['max_supply'] ?? null,
    ];
    $algo = $data['hashing_algorithm'] ?? null;
    $links = $data['links'] ?? null;

    $upsert->execute([
        ':coin_id' => (int)$coin['id'],
        ':description_html' => $desc,
        ':homepage' => $homepage,
        ':explorers_json' => json_encode($explorers),
        ':community_json' => json_encode($community),
        ':dev_json' => json_encode($dev),
        ':supply_json' => json_encode($supply),
        ':hashing_algo' => $algo,
        ':links_json' => json_encode($links),
    ]);

    // Invalidate snapshot
    // Fetch slug to build route
    $slugStmt = $pdo->prepare('SELECT slug FROM coins WHERE id = :id');
    $slugStmt->execute([':id' => (int)$coin['id']]);
    $slug = $slugStmt->fetchColumn();
    if ($slug) Snapshot::invalidate('/coins/' . $slug);
}

echo "metadata_refresh done\n";
