<?php
// OHLC sync per coin and interval (daily as example)
require __DIR__ . '/../../app/bootstrap.php';
require_once __DIR__ . '/../lib/http.php';
require_once __DIR__ . '/../lib/lock.php';

use App\Database;
use App\Config;
use Scripts\Lib\Http;
use Scripts\Lib\HttpException;
use Scripts\Lib\Lock;
use App\Snapshot;

if (php_sapi_name() !== 'cli') { echo "Run from CLI\n"; exit(1); }
set_time_limit(0);
ini_set('memory_limit','512M');

$lock = new Lock('ohlc_sync');
if (!$lock->acquire()) { echo "Another ohlc_sync is running.\n"; exit(0); }

$pdo = Database::pdo();
$base = rtrim(Config::get('coingecko.base'), '/');
$currency = strtolower(Config::get('app.currency_default','USD'));

// Interval selection via argv: default 1d. Options: 1d,4h,1h
$interval = '1d';
foreach ($argv as $arg) { if (in_array($arg, ['1d','4h','1h'], true)) { $interval = $arg; break; } }

$table = match ($interval) {
    '1h' => 'coin_ohlc_1h',
    '4h' => 'coin_ohlc_4h',
    default => 'coin_ohlc_1d',
};

$coinsStmt = $pdo->query("SELECT id, cg_id FROM coins WHERE is_active = 1 AND cg_id IS NOT NULL ORDER BY rank IS NULL, rank ASC LIMIT 200");
$coins = $coinsStmt->fetchAll();

$ins = $pdo->prepare("INSERT IGNORE INTO {$table} (coin_id, currency, t, o, h, l, c, v) VALUES (:coin_id, :currency, :t, :o, :h, :l, :c, :v)");

foreach ($coins as $coin) {
    $cgId = $coin['cg_id'];
    // CoinGecko OHLC endpoint returns candles; use days=max (for daily) or smaller windows for intraday as available.
    $days = $interval === '1d' ? 'max' : '1';
    $url = $base . "/coins/{$cgId}/ohlc?vs_currency={$currency}&days={$days}";
    echo "Fetching OHLC for {$cgId}...\n";
    try {
        // Response format: [ [timestamp, open, high, low, close], ...]
        $data = Http::getJson($url, (int)Config::get('coingecko.timeout_sec', 20));
    } catch (HttpException $e) {
        if ($e->getStatus() === 429 && $e->getRetryAfter()) { sleep($e->getRetryAfter()); }
        fwrite(STDERR, "HTTP error for {$cgId}: " . $e->getMessage() . "\n");
        continue;
    } catch (\Throwable $e) {
        fwrite(STDERR, "HTTP error for {$cgId}: " . $e->getMessage() . "\n");
        continue;
    }
    if (!is_array($data)) continue;

    $pdo->beginTransaction();
    try {
        foreach ($data as $row) {
            if (!is_array($row) || count($row) < 5) continue;
            $ts = (int)round(($row[0] ?? 0)/1000);
            $t = $interval === '1d' ? date('Y-m-d 00:00:00', $ts) : date('Y-m-d H:00:00', $ts);
            $ins->execute([
                ':coin_id' => (int)$coin['id'],
                ':currency' => $currency,
                ':t' => $t,
                ':o' => $row[1] ?? null,
                ':h' => $row[2] ?? null,
                ':l' => $row[3] ?? null,
                ':c' => $row[4] ?? null,
                ':v' => null, // volume not provided by /ohlc
            ]);
        }
        $pdo->commit();
    } catch (\Throwable $e) {
        $pdo->rollBack();
        fwrite(STDERR, "DB error for {$cgId}: " . $e->getMessage() . "\n");
    }

    // Invalidate snapshot for this coin
    $slugStmt = $pdo->prepare('SELECT slug FROM coins WHERE id = :id');
    $slugStmt->execute([':id' => (int)$coin['id']]);
    $slug = $slugStmt->fetchColumn();
    if ($slug) Snapshot::invalidate('/coins/' . $slug);
}

echo "ohlc_sync done\n";
