-- MySQL 8 schema (core tables for MVP)
-- Convention: snake_case, id BIGINT PK, timestamps

CREATE TABLE IF NOT EXISTS coins (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  cg_id VARCHAR(100) NOT NULL,
  symbol VARCHAR(20) NOT NULL,
  name VARCHAR(120) NOT NULL,
  slug VARCHAR(140) NOT NULL,
  image_thumb VARCHAR(255) NULL,
  image_small VARCHAR(255) NULL,
  category_primary VARCHAR(100) NULL,
  platform_primary VARCHAR(100) NULL,
  contract_address VARCHAR(120) NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  rank INT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uq_coins_cg (cg_id),
  KEY idx_coins_slug (slug),
  KEY idx_coins_rank (rank)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS coin_metadata (
  coin_id BIGINT UNSIGNED NOT NULL,
  description_html MEDIUMTEXT NULL,
  homepage VARCHAR(255) NULL,
  explorers_json JSON NULL,
  community_json JSON NULL,
  dev_json JSON NULL,
  supply_json JSON NULL,
  ath DECIMAL(24,8) NULL,
  atl DECIMAL(24,8) NULL,
  ath_date DATETIME NULL,
  atl_date DATETIME NULL,
  hashing_algo VARCHAR(100) NULL,
  links_json JSON NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (coin_id),
  CONSTRAINT fk_cm_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS coin_market_snapshots (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  coin_id BIGINT UNSIGNED NOT NULL,
  currency VARCHAR(10) NOT NULL,
  price DECIMAL(24,8) NULL,
  market_cap DECIMAL(28,2) NULL,
  volume_24h DECIMAL(28,2) NULL,
  change_1h DECIMAL(9,4) NULL,
  change_24h DECIMAL(9,4) NULL,
  change_7d DECIMAL(9,4) NULL,
  change_30d DECIMAL(9,4) NULL,
  ath_drawdown DECIMAL(9,4) NULL,
  timestamp DATETIME NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  UNIQUE KEY uq_cms_uni (coin_id, currency, timestamp),
  KEY idx_cms_coin (coin_id),
  KEY idx_cms_cur (currency),
  KEY idx_cms_coin_cur_ts (coin_id, currency, timestamp),
  KEY idx_cms_mcap (market_cap),
  KEY idx_cms_vol (volume_24h),
  KEY idx_cms_ch24 (change_24h),
  KEY idx_cms_ts (timestamp),
  CONSTRAINT fk_cms_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Optional: daily OHLC table (skeleton)
CREATE TABLE IF NOT EXISTS coin_ohlc_1d (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  coin_id BIGINT UNSIGNED NOT NULL,
  currency VARCHAR(10) NOT NULL,
  t DATETIME NOT NULL,
  o DECIMAL(24,8) NULL,
  h DECIMAL(24,8) NULL,
  l DECIMAL(24,8) NULL,
  c DECIMAL(24,8) NULL,
  v DECIMAL(28,2) NULL,
  UNIQUE KEY uq_ohlc1d (coin_id, currency, t),
  KEY idx_ohlc_coin (coin_id),
  KEY idx_ohlc_cur (currency),
  KEY idx_ohlc_t (t),
  CONSTRAINT fk_ohlc_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Intraday OHLC tables
CREATE TABLE IF NOT EXISTS coin_ohlc_4h (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  coin_id BIGINT UNSIGNED NOT NULL,
  currency VARCHAR(10) NOT NULL,
  t DATETIME NOT NULL,
  o DECIMAL(24,8) NULL,
  h DECIMAL(24,8) NULL,
  l DECIMAL(24,8) NULL,
  c DECIMAL(24,8) NULL,
  v DECIMAL(28,2) NULL,
  UNIQUE KEY uq_ohlc4h (coin_id, currency, t),
  KEY idx_ohlc4h_coin (coin_id),
  KEY idx_ohlc4h_cur (currency),
  KEY idx_ohlc4h_t (t),
  CONSTRAINT fk_ohlc4h_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS coin_ohlc_1h (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  coin_id BIGINT UNSIGNED NOT NULL,
  currency VARCHAR(10) NOT NULL,
  t DATETIME NOT NULL,
  o DECIMAL(24,8) NULL,
  h DECIMAL(24,8) NULL,
  l DECIMAL(24,8) NULL,
  c DECIMAL(24,8) NULL,
  v DECIMAL(28,2) NULL,
  UNIQUE KEY uq_ohlc1h (coin_id, currency, t),
  KEY idx_ohlc1h_coin (coin_id),
  KEY idx_ohlc1h_cur (currency),
  KEY idx_ohlc1h_t (t),
  CONSTRAINT fk_ohlc1h_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Users & Auth
CREATE TABLE IF NOT EXISTS users (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  email VARCHAR(190) NOT NULL,
  password_hash VARCHAR(255) NULL,
  name VARCHAR(120) NULL,
  avatar VARCHAR(255) NULL,
  oauth_google_id VARCHAR(190) NULL,
  currency_pref VARCHAR(10) NULL,
  timezone VARCHAR(64) NULL,
  email_verified TINYINT(1) NOT NULL DEFAULT 0,
  role VARCHAR(32) NOT NULL DEFAULT 'user',
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uq_users_email (email),
  UNIQUE KEY uq_users_google (oauth_google_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS sessions (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  user_id BIGINT UNSIGNED NOT NULL,
  session_token VARCHAR(190) NOT NULL,
  ip VARCHAR(64) NULL,
  ua VARCHAR(255) NULL,
  expires_at DATETIME NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  KEY idx_sessions_user (user_id),
  UNIQUE KEY uq_sessions_token (session_token),
  CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Watchlists
CREATE TABLE IF NOT EXISTS watchlists (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  user_id BIGINT UNSIGNED NOT NULL,
  name VARCHAR(120) NOT NULL,
  slug VARCHAR(160) NOT NULL,
  visibility ENUM('public','unlisted','private') NOT NULL DEFAULT 'private',
  is_default TINYINT(1) NOT NULL DEFAULT 0,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY idx_w_user (user_id),
  UNIQUE KEY uq_w_user_slug (user_id, slug),
  CONSTRAINT fk_w_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS watchlist_items (
  watchlist_id BIGINT UNSIGNED NOT NULL,
  coin_id BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY (watchlist_id, coin_id),
  KEY idx_wi_coin (coin_id),
  CONSTRAINT fk_wi_w FOREIGN KEY (watchlist_id) REFERENCES watchlists(id) ON DELETE CASCADE,
  CONSTRAINT fk_wi_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Screens (basic)
CREATE TABLE IF NOT EXISTS screens (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  user_id BIGINT UNSIGNED NOT NULL,
  name VARCHAR(160) NOT NULL,
  slug VARCHAR(180) NOT NULL,
  visibility ENUM('public','unlisted','private') NOT NULL DEFAULT 'private',
  definition_json JSON NOT NULL,
  sort VARCHAR(64) NULL,
  `limit` INT NULL,
  currency VARCHAR(10) NULL,
  fork_of_screen_id BIGINT UNSIGNED NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uq_screens_slug (slug),
  KEY idx_s_user (user_id),
  CONSTRAINT fk_s_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS screen_stars (
  user_id BIGINT UNSIGNED NOT NULL,
  screen_id BIGINT UNSIGNED NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (user_id, screen_id),
  KEY idx_ss_screen (screen_id),
  CONSTRAINT fk_ss_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  CONSTRAINT fk_ss_screen FOREIGN KEY (screen_id) REFERENCES screens(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Alerts (skeleton)
CREATE TABLE IF NOT EXISTS alerts (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  user_id BIGINT UNSIGNED NOT NULL,
  coin_id BIGINT UNSIGNED NULL,
  type ENUM('price','pct','volume','screen_match') NOT NULL,
  condition_json JSON NOT NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  last_triggered_at DATETIME NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  KEY idx_a_user (user_id),
  KEY idx_a_coin (coin_id),
  CONSTRAINT fk_a_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  CONSTRAINT fk_a_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS notifications (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  user_id BIGINT UNSIGNED NOT NULL,
  title VARCHAR(160) NOT NULL,
  body TEXT NOT NULL,
  channel ENUM('email','webpush') NOT NULL DEFAULT 'email',
  status VARCHAR(32) NOT NULL DEFAULT 'queued',
  error VARCHAR(255) NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  KEY idx_n_user (user_id),
  CONSTRAINT fk_n_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
-- Admin settings (key/value JSON storage)
CREATE TABLE IF NOT EXISTS admin_settings (
  `key` VARCHAR(100) NOT NULL PRIMARY KEY,
  `value_json` JSON NULL,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Cache store (generic key/value with expiry)
CREATE TABLE IF NOT EXISTS cache_store (
  `key` VARCHAR(190) NOT NULL PRIMARY KEY,
  `value` MEDIUMTEXT NULL,
  `expires_at` DATETIME NULL,
  KEY idx_cache_exp (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Login throttling
CREATE TABLE IF NOT EXISTS login_attempts (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  email VARCHAR(190) NULL,
  ip VARCHAR(64) NULL,
  attempts INT NOT NULL DEFAULT 0,
  window_start DATETIME NOT NULL,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY idx_la_email (email),
  KEY idx_la_ip (ip)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Jobs & worker
CREATE TABLE IF NOT EXISTS jobs (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  type VARCHAR(100) NOT NULL,
  payload_json JSON NULL,
  status ENUM('queued','running','done','error') NOT NULL DEFAULT 'queued',
  retries INT NOT NULL DEFAULT 0,
  not_before DATETIME NULL,
  last_error VARCHAR(255) NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY idx_jobs_status (status),
  KEY idx_jobs_not_before (not_before)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Exchanges
CREATE TABLE IF NOT EXISTS exchanges (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  cg_id VARCHAR(100) NOT NULL,
  name VARCHAR(160) NOT NULL,
  slug VARCHAR(180) NOT NULL,
  year_established INT NULL,
  country VARCHAR(80) NULL,
  trust_score INT NULL,
  url VARCHAR(255) NULL,
  logo VARCHAR(255) NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uq_ex_cg (cg_id),
  KEY idx_ex_slug (slug),
  KEY idx_ex_trust (trust_score)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS exchange_markets (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  exchange_id BIGINT UNSIGNED NOT NULL,
  base_symbol VARCHAR(40) NOT NULL,
  quote_symbol VARCHAR(40) NOT NULL,
  pair VARCHAR(90) NOT NULL,
  last_price DECIMAL(24,8) NULL,
  volume_24h DECIMAL(28,2) NULL,
  is_spot TINYINT(1) NOT NULL DEFAULT 1,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY idx_em_ex (exchange_id),
  KEY idx_em_vol (volume_24h),
  CONSTRAINT fk_em_ex FOREIGN KEY (exchange_id) REFERENCES exchanges(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Categories
CREATE TABLE IF NOT EXISTS categories (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
  cg_id VARCHAR(100) NOT NULL,
  name VARCHAR(160) NOT NULL,
  slug VARCHAR(180) NOT NULL,
  market_cap DECIMAL(28,2) NULL,
  volume_24h DECIMAL(28,2) NULL,
  num_coins INT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uq_cat_cg (cg_id),
  UNIQUE KEY uq_cat_slug (slug)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS coin_categories (
  coin_id BIGINT UNSIGNED NOT NULL,
  category_id BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY (coin_id, category_id),
  KEY idx_cc_cat (category_id),
  CONSTRAINT fk_cc_coin FOREIGN KEY (coin_id) REFERENCES coins(id) ON DELETE CASCADE,
  CONSTRAINT fk_cc_cat FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
